/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 */

#ifndef GUI_1CHANNEL_H
#define GUI_1CHANNEL_H

#include "config_handler.h"
#include "loggingclass.h"
#include "numpad.h"
#include "rf_system.h"
#include "rf_system_1channel.h"
#include <QButtonGroup>
#include <QMainWindow>
#include <QMap>
#include <QPushButton>
#include <QTcpServer>
#include <QTcpSocket>
#include <QThread>
#include <QTimer>
#include <QtSerialPort/QSerialPort>
#include <QListWidgetItem>

namespace Ui {
class GUI_1channel;
}

class GUI_1channel : public QMainWindow
{
	Q_OBJECT

public:
	explicit GUI_1channel(QWidget *parent = 0);
	~GUI_1channel();


signals:
	//MISCELLANEOUS
	void signal_channelInit(int intrasys_num);
	void signal_timerStart(int intrasys_num, int ms);
	void signal_timerStop(int intrasys_num);

	void signal_RF_enable(int intrasys_num, bool enable);
	void signal_execute_reset_SGx(int intrasys_num);
	void signal_execute_error_clear(int intrasys_num);
	void signal_request_PA_power_update(int intrasys_num);

	void signal_setClock_source(int intrasys_num, int mode);			//0 = Standalone, 1 = Master, 2 = Slave (end), 3 = Slave (inline)
	void signal_setPWM_triggering_mode(int intrasys_num, int mode);		//1 = Free-running, 2 = Triggered Master, 3 = Triggered Slave
	void signal_setExternal_triggering_enable(int intrasys_num, bool enable);
	void signal_set_power_control_mode(int intrasys_num, int mode);
	void signal_set_error_clearing_enable(int intrasys_num, bool enable);

	//CW
	void signal_setVGA_attenuation(int intrasys_num, double attenuation);
	void signal_setIQMod_magnitude(int intrasys_num, double magnitude);
	void signal_setPowerWatt(int intrasys_num, double power_watt);
	void signal_setPowerdBm(int intrasys_num, double power_dbm);
	void signal_setFrequency(int intrasys_num, double frequency);
	void signal_setPhase(int intrasys_num, double phase);
	void signal_setOffsetdB(int intrasys_num, double offset_dB);

	//NChannel / Splitter
	void signal_setPhase_NChannel(int intrasys_num, int subchannel, double phase);
	void signal_setMagnitude_NChannel(int intrasys_num, int subchannel, double magnitude);
	void signal_setAttenuation_NChannel(int intrasys_num, int subchannel, double attenuation);

	//PWM
	void signal_setPWM_enable(int intrasys_num, bool enable);
	void signal_setPWM_frequency(int intrasys_num, double pwm_frequency);
	void signal_setPWM_duty_cycle(int intrasys_num, double pwm_duty_cycle);

	//DLL
	void signal_setDLL_enable(int intrasys_num, bool enable);
	void signal_setDLL_frequency_limit_upper(int intrasys_num, double all_frequency_limit_upper);
	void signal_setDLL_frequency_limit_lower(int intrasys_num, double all_frequency_limit_lower);
	void signal_setDLL_start_frequency(int intrasys_num, double start_frequency);
	void signal_setDLL_step_frequency(int intrasys_num, double step_frequency);
	void signal_setDLL_threshold (int intrasys_num, double dll_threshold);
	void signal_setDLL_main_delay(int intrasys_num, double main_delay);

	//ALL
	void signal_setALL_enable(int intrasys_num, bool enable);
	void signal_setALL_frequency_limit_upper(int intrasys_num, double all_frequency_limit_upper);
	void signal_setALL_frequency_limit_lower(int intrasys_num, double all_frequency_limit_lower);
	void signal_setALL_threshold(int intrasys_num, double all_threshold);

	//DVGA
	void signal_setDVGA_amplifier_forward_enable(int intrasys_num, bool enable);
	void signal_setDVGA_amplifier_reflected_enable(int intrasys_num, bool enable);
	void signal_setDVGA_attentuation_forward(int intrasys_num, double attenuation_fwd);
	void signal_setDVGA_attentuation_reflected(int intrasys_num, double attenuation_rfl);

	//SWEEP
	void signal_setSWP_power_dbm(int intrasys_num, double power_dbm);
	void signal_setSWP_power_watt(int intrasys_num, double power_watt);
	void signal_setSWP_start_frequency(int intrasys_num, double start_frequency);
	void signal_setSWP_stop_frequency(int intrasys_num, double stop_frequency);
	void signal_setSWP_step_frequency(int intrasys_num, double step_frequency);
	void signal_execute_SWP(int intrasys_num);

	//PSU
	void signal_setPSU_enable(int intrasys_num, bool enable);
	void signal_setPSU_voltage(int intrasys_num, double val);
	void signal_setPSU_current_limit(int intrasys_num, double val);

	//RCM
	void signal_remote_command_mode_set(RCM_MODE mode);

	//MODBUS
	void signal_set_modbus_enable(bool enable);

	//EtherCAT
	void signal_set_ethercat_enable(bool enable);


/**********************************************************************************************************************************************************************************
 * SLOTS
 * *******************************************************************************************************************************************************************************/

private slots:
	//Miscellaneous Functions
	void handler_execute_restart_program();
	void handler_notification_get(int intrasys_num, QString notification);
	void handler_channel_working(int intrasys_number, bool enabled);
	void handler_channel_initialization_complete(int intrasys_num, bool success);
	void handler_splitter_configuration(int intrasys_num, int splitter_mode, int splitter_channel_count);
	void handler_firmware_version_get(int intrasys_num, int channel_num, QString version);
	void handler_board_identity_get(int intrasys_num, QString manufacturer, QString model, QString serialnumber);
	void handler_hardware_identity_get(int intrasys_num, int identity_number, QString identity_string);
	void handler_PA_type_get(int intrasys_num, int PA_type);
	void handler_RF_enable_get(int intrasys_num, bool RF_enabled);
	void handler_PA_power_readings(int intrasys_num, double PA_power_fwd_dbm, double PA_power_rfl_dbm, double PA_s11, double PA_power_fwd_watt, double PA_power_rfl_watt, double PA_s11_reflection);
	void handler_PA_power_readings_NChannel(int intrasys_num, int subchannel, double PA_power_fwd_dbm, double PA_power_rfl_dbm, double PA_s11, double PA_power_fwd_watt, double PA_power_rfl_watt, double PA_s11_reflection);
	void handler_error_get(int intrasys_num, quint64 error, QStringList error_messages);
	void handler_reset_detected(int intrasys_num, int channel_number);
	void handler_temperature_PA_get(int intrasys_num, double temperature_PA);
	void handler_temperature_SG_get(int intrasys_num, double temperature_SG);
	void handler_clock_source_get(int intrasys_num, int mode);
	void handler_PWM_triggering_mode_get(int intrasys_num, int mode);
	void handler_power_control_mode_get(int intrasys_num, int mode);

	//Feature availability management
	void handler_availability_amplifier_mode(int intrasys_num, bool visible, bool available);
	void handler_availability_external_triggering(int intrasys_num, bool visible, bool available);
	void handler_availability_PWM(int intrasys_num, bool visible, bool available);
	void handler_availability_ALL(int intrasys_num, bool visible, bool available);
	void handler_availability_DLL(int intrasys_num, bool visible, bool available);
	void handler_availability_sweep(int intrasys_num, bool visible, bool available);
	void handler_availability_PSU_enable(int intrasys_num, bool visible, bool available);

	//CW
	void handler_VGA_attenuation_get(int intrasys_num, double attenuation);
	void handler_IQMod_magnitude_get(int intrasys_num, double magnitude);
	void handler_power_get(int intrasys_num, double power_dbm, double power_watt);
	void handler_frequency_get(int intrasys_num, double frequency_mhz);
	void handler_offset_db_get(int intrasys_num, double offset_db);
	void handler_phase_get(int intrasys_num, double phase_degrees);
	void handler_CW_enable_get(int intrasys_num, bool enable);

	//NChannel / Splitter
	void handler_phase_NChannel_get(int intrasys_num, int subchannel, double phase_degrees);
	void handler_attenuation_NChannel_get(int intrasys_num, int subchannel, double attenuation);
	void handler_magnitude_NChannel_get(int intrasys_num, int subchannel, double magnitude);

	//PWM
	void handler_PWM_settings_get(int intrasys_num, double frequency_hz, int duty_cycle);
	void handler_external_triggering_enable_get(int intrasys_num, bool enable);

	//DLL
	void handler_DLL_enable_get(int intrasys_num, bool enable);
	void handler_DLL_settings_get(int intrasys_num, double frequency_lower, double frequency_upper, double frequency_start, double frequency_step, double threshold, double main_delay);

	//ALL
	void handler_ALL_compatible_get(int intrasys_num, bool compatibility);
	void handler_ALL_enable_get(int intrasys_num, bool enable);
	void handler_ALL_settings_get(int intrasys_num, double frequency_lower_voltage, double frequency_upper_voltage, double threshold_voltage);

	//DVGA
	void handler_DVGA_compatible_get(int intrasys_num, bool compatibility);
	void handler_DVGA_forward_settings_get(int intrasys_num, bool amplifier_enable, double attenuation);
	void handler_DVGA_reflected_settings_get(int intrasys_num, bool amplifier_enable, double attenuation);

	//SWP
	void handler_SWP_settings_get(int intrasys_num, double frequency_start, double frequency_stop, double frequency_step, double power_dbm, double power_watt);
	void handler_SWP_measurement_get(int intrasys_num, QString SWP_raw_data);

	//PSU
	void handler_PSU_voltage_setpoint_get(int intrasys_num, double val);
	void handler_PSU_current_limit_get(int intrasys_num, double val);
	void handler_PSU_enable_combined_get(int intrasys_num, bool enable);
	void handler_PSU_IU_get(int intrasys_num, int psu_num, double voltage, double current, double power);
	void handler_PSU_power_efficiency_get(int intrasys_num, double efficiency);
	void handler_PSU_dissipation_get(int intrasys_num, double dissipation);
	void handler_PSU_voltage_measurement_get(int intrasys_num, double PSU_voltage_combined);
	void handler_PSU_current_measurement_get(int intrasys_num, double PSU_current_combined);
	void PSU_reset();	//workaround function / to be deleted

	//Data Logging
	void handler_datalogging_enable_get(bool enable);
	void handler_datalogging_storage_sufficient(bool valid);
	void handler_datalogging_log_deleted(bool success, QString filename);
	void handler_datalogging(bool enable);

	//Notification bar
	void show_notification(QString message, int time = 8000);
	void close_notification();
	void show_error(QString message);
	void close_error();

	//Miscellaneous buttons
	void on_pushButton_RF_enable_1_clicked();
	void on_pushButton_unit_power_reflected_1_clicked();
	void on_pushButton_unit_power_forward_1_clicked();
	void on_pushButton_unit_S11_1_clicked();
	void on_pushButton_CW_enable_1_clicked();
	void on_pushButton_PWM_enable_1_clicked();
	void on_pushButton_ALL_enable_1_clicked();
	void on_pushButton_DLL_enable_1_clicked();

	//Numpad Buttons
	void on_numpadButton_0_clicked();
	void on_numpadButton_1_clicked();
	void on_numpadButton_2_clicked();
	void on_numpadButton_3_clicked();
	void on_numpadButton_4_clicked();
	void on_numpadButton_5_clicked();
	void on_numpadButton_6_clicked();
	void on_numpadButton_7_clicked();
	void on_numpadButton_8_clicked();
	void on_numpadButton_9_clicked();
	void on_numpadButton_period_clicked();
	void on_numpadButton_arrow_left_clicked();
	void on_numpadButton_arrow_right_clicked();
	void on_numpadButton_plus_clicked();
	void on_numpadButton_minus_clicked();
	void on_numpadButton_sign_clicked();
	void on_numpadButton_backspace_clicked();
	void on_numpadButton_clear_all_clicked();
	void on_numpadButton_ok_clicked();

	//Numpad Functions
	void numpad_value_confirmed_handler(QPushButton *button, double *num);

	//Menu Buttons
	void on_menu_home_Button_clicked();
	void on_menu_splitter_Button_clicked();
	void on_menu_ALL_Button_clicked();
	void on_menu_DLL_Button_clicked();
	void on_menu_PSU_Button_clicked();
	void on_menu_sweep_Button_clicked();
	void on_menu_help_Button_clicked();
	void on_menu_settings_Button_clicked();
	void on_menu_power_Button_clicked();

	//Home Menu
	void on_pushButton_IQMod_magnitude_1_clicked();
	void on_pushButton_VGA_attenuation_1_clicked();
	void on_pushButton_power_dbm_1_clicked();
	void on_pushButton_power_watt_1_clicked();
	void on_pushButton_frequency_1_clicked();
	void on_pushButton_power_offset_dB_1_clicked();
	void on_pushButton_phase_1_clicked();

	void on_pushButton_PWM_frequency_1_clicked();
	void on_pushButton_PWM_duty_cycle_1_clicked();
	void on_pushButton_ALL_threshold_1_clicked();
	void on_pushButton_DLL_threshold_1_clicked();
	void on_pushButton_DLL_step_frequency_1_clicked();

	//NChannel / Splitter Menu
	void on_pushButton_splitter_phase_CH1_clicked();
	void on_pushButton_splitter_phase_CH2_clicked();
	void on_pushButton_splitter_phase_CH3_clicked();
	void on_pushButton_splitter_phase_CH4_clicked();
	void on_pushButton_splitter_phase_CH5_clicked();
	void on_pushButton_splitter_phase_CH6_clicked();
	void on_pushButton_splitter_phase_CH7_clicked();
	void on_pushButton_splitter_phase_CH8_clicked();
	void on_pushButton_splitter_attenuation_CH1_clicked();
	void on_pushButton_splitter_attenuation_CH2_clicked();
	void on_pushButton_splitter_attenuation_CH3_clicked();
	void on_pushButton_splitter_attenuation_CH4_clicked();
	void on_pushButton_splitter_attenuation_CH5_clicked();
	void on_pushButton_splitter_attenuation_CH6_clicked();
	void on_pushButton_splitter_attenuation_CH7_clicked();
	void on_pushButton_splitter_attenuation_CH8_clicked();
	void on_pushButton_splitter_magnitude_CH1_clicked();
	void on_pushButton_splitter_magnitude_CH2_clicked();
	void on_pushButton_splitter_magnitude_CH3_clicked();
	void on_pushButton_splitter_magnitude_CH4_clicked();
	void on_pushButton_splitter_magnitude_CH5_clicked();
	void on_pushButton_splitter_magnitude_CH6_clicked();
	void on_pushButton_splitter_magnitude_CH7_clicked();
	void on_pushButton_splitter_magnitude_CH8_clicked();

	//ALL Menu
	void on_pushButton_ALL_frequency_limit_upper_1_clicked();
	void on_pushButton_ALL_frequency_limit_lower_1_clicked();
	void on_pushButton_ALL_threshold_2_clicked();

	void on_pushButton_ALL_reference_open_clicked();
	void on_pushButton_ALL_reference_back_clicked();	//back button the reference page

	void on_pushButton_DVGA_attenuation_forward_1_clicked();
	void on_pushButton_DVGA_attenuation_reflected_1_clicked();
	void on_pushButton_DVGA_amplifier_forward_1_clicked();
	void on_pushButton_DVGA_amplifier_reflected_1_clicked();

	//DLL Menu
	void on_pushButton_DLL_frequency_limit_upper_1_clicked();
	void on_pushButton_DLL_frequency_limit_lower_1_clicked();
	void on_pushButton_DLL_start_frequency_1_clicked();
	void on_pushButton_DLL_step_frequency_2_clicked();
	void on_pushButton_DLL_threshold_2_clicked();
	void on_pushButton_DLL_main_delay_1_clicked();
	void on_pushButton_power_dbm_2_clicked();
	void on_pushButton_power_watt_2_clicked();

	//Sweep Menu
	void on_pushButton_SWP_unit_clicked();
	void on_pushButton_SWP_execute_2_clicked();
	void on_pushButton_SWP_execute_1_clicked();
	void on_pushButton_SWP_back_clicked();
	void on_pushButton_SWP_set_best_frequency_1_clicked();

	void on_pushButton_SWP_start_frequency_1_clicked();
	void on_pushButton_SWP_stop_frequency_1_clicked();
	void on_pushButton_SWP_step_frequency_1_clicked();
	void on_pushButton_SWP_power_dbm_1_clicked();
	void on_pushButton_SWP_power_watt_1_clicked();

	//PSU Menu
	void on_pushButton_PSU_voltage_1_clicked();
	void on_pushButton_PSU_current_limit_1_clicked();
	void on_pushButton_PSU_enable_1_clicked(bool checked);

	//Settings Menu
	void on_pushButton_power_control_normal_1_clicked();			//Regular Power Control (auto-gain ON)
	void on_pushButton_power_control_feedforward_1_clicked();		//Feed Forward Power Control (auto-gain OFF)
	void on_pushButton_power_control_analog_1_clicked();			//Analog Input Power Control (auto-gain OFF)
	void on_pushButton_power_control_amplifier_1_clicked();			//Amplifier Power Control Mode
	void on_pushButton_power_control_PID_1_clicked();				//PID Power Control

	void on_pushButton_clock_source_standalone_1_clicked();			//Coherent mode: Off
	void on_pushButton_clock_source_master_1_clicked();				//Coherent mode: LVDS Master
	void on_pushButton_clock_source_slave_inline_1_clicked();		//Coherent mode: LVDS Slave (Inline)
	void on_pushButton_clock_source_slave_end_1_clicked();			//Coherent mode: LVDS Slave (End of line) - not used much...
	void on_pushButton_clock_source_AUX_master_1_clicked();			//Coherent mode: AUX Master
	void on_pushButton_clock_source_AUX_slave_end_1_clicked();		//Coherent mode: AUX Slave (End of line)

	void on_pushButton_external_triggering_OFF_1_clicked();			//External Triggering OFF
	void on_pushButton_external_triggering_ON_1_clicked();			//External Triggering ON

	void on_pushButton_PWM_triggering_free_running_1_clicked();		//Free-running
	void on_pushButton_PWM_triggering_master_1_clicked();			//Triggered Master
	void on_pushButton_PWM_triggering_slave_1_clicked();			//Triggered Slave

	void on_pushButton_data_logging_OFF_1_clicked();
	void on_pushButton_data_logging_ON_1_clicked();
//	void on_pushButton_data_logging_ON_1_toggled(bool checked);

	void on_pushButton_remote_command_OFF_1_clicked();				//RCM OFF
	void on_pushButton_remote_command_USB_1_clicked();				//RCM USB LIVE
	void on_pushButton_remote_command_USB_blind_1_clicked();		//RCM USB BLIND
	void on_pushButton_remote_command_RS232_1_clicked();			//RCM RS232 LIVE
	void on_pushButton_remote_command_RS232_blind_1_clicked();		//RCM RS232 BLIND
	void on_pushButton_remote_command_TCP_1_clicked();				//RCM TCP LIVE
	void on_pushButton_remote_command_TCP_blind_1_clicked();		//RCM TCP BLIND

    //Remote Command mode
	void handler_RCM_notification(QString string);
	void handler_remote_command_mode_get(RCM_MODE mode);
	void handler_RCM_error_clearing(int intrasys_num, bool enable);

	void RCM_controls_enable(bool enable);

	//Modbus mode
	void on_pushButton_modbus_OFF_1_clicked();					//Modbus OFF
	void on_pushButton_modbus_TCP_slave_1_clicked();			//Modbus TCP Slave
	void handler_modbus_connected_enable(bool enable, QString address, int port, QString error);

	//EtherCAT
	void on_pushButton_etherCAT_OFF_1_clicked();				//EtherCAT OFF
	void on_pushButton_etherCAT_ON_1_clicked();					//EtherCAT ON

	//Help menu
	void on_listWidget_currentItemChanged(QListWidgetItem *current, QListWidgetItem *previous);

	//Power menu
	void on_pushButton_clear_errors_1_clicked();
	void on_pushButton_reset_SGx_1_clicked();
	void on_pushButton_restart_program_1_clicked();
	void on_pushButton_exit_program_1_clicked();

	//
	// TODO:
	// Fix these names!
	//
	void on_Exit_Yes_button_clicked();
	void on_Exit_No_button_clicked();

/**********************************************************************************************************************************************************************************
 * Variables and functions
 * *******************************************************************************************************************************************************************************/

private:
	Ui::GUI_1channel *ui;

	RF_System_1channel *RF_system_constructor;
	ConfigHandler *config;
	double numpad_value = 0;

	QString SG_firmware_version = "";
	QString SG_manufacturer = "";
	QString SG_board_model = "";
	QString SG_serial_number = "";
	QString hardware_identity_string = "";
	int hardware_identity_number = -1;
	int PA_type = -1;

	/* Remote Command Mode */
	void handler_RCM_mode_startup();
	bool RCM_supported = false;
	RCM_MODE remote_command_mode = RCM_MODE::RCM_MODE_OFF;
	QString ipAddress;				//IP Address of this system

	/* Modbus Mode */
	void handler_modbus_mode();

	/* EtherCAT Mode */
	void handler_ethercat_mode();

	/* Datalogging */
	LoggingClass *datalogger;

	/* SWEEP variables */
	void SWP_enable_GUI(bool enable);
	void SWP_calculate_best_match(QVector<double> freq_data, QVector<double> s11_dbm_data);
	void SWP_draw_plot(QString SWP_raw_data);
	double best_S11_match_frequency = 0;
	double best_S11_match_dB = __INT_MAX__;
	QString SWP_mode = "dbm";

	bool sweep_visible = false;

	/* Coherency Mode */
	bool coherency_supported = false;

	/* CORE */
	QString channel_select = "1";
	StatusCheck *status_checker;

	QTimer *notification_timer;

	/* Splitter Page */
	QList<QFrame*> frame_list;
	QList<QPushButton*> splitter_button_magnitude_list;
	QList<QPushButton*> splitter_button_attenuation_list;
	QList<QPushButton*> splitter_button_phase_list;
	QList<QLabel*> splitter_label_magnitude_list;
	QList<QLabel*> splitter_label_attenuation_list;
	QList<QLabel*> splitter_label_phase_list;

	/* Help Page */
	QListWidgetItem listWidgetItem_system;
	QListWidgetItem listWidgetItem_about;
	QListWidgetItem listWidgetItem_license;
	QString system_string = "";
	void update_listWidgetItem_system();

	/* Startup functions */
	void firmware_version_check(QString firmware_version);
	std::array<int,4> firmware_version_requirement;
	QString	toolkit_GUI_version;

	/* Automatic Error Clearing */
	bool RCM_error_clearing = true;
	void configure_auto_error_clearing();

	/* Feature Availability and Visibility */
	// External Triggering
	bool visibility_ext_triggering = false;
	bool availability_ext_triggering = false;
	void configure_availability_external_triggering();

	// Power Control Modes
	bool availability_amplifier_mode = false;
	bool visibility_amplifier_mode = false;
	void configure_availability_amplifier_mode();

	// Other
	void show_DLL_settings(bool input);
	void show_ALL_settings(bool input);
	void show_PWM_settings(bool input);
	void show_autogain_controls(int intrasys_num, bool enable, bool visible);
	void show_sweep_power_controls(int intrasys_num, bool enable, bool visible);
	void show_sweep_button();
	void show_amplifier_mode_controls(int intrasys_num, bool enable);
	void show_GCS_controls(int intrasys_num, bool enable, bool visible);
	void show_MCS_controls(int intrasys_num, bool enable, bool visible);

	/* Miscellaneous */
	void set_power_reading_units_watt(bool mode);
	void menuButton_manager(QPushButton *menuButton_pointer);		//controls behaviour of menu buttons / page hopping

	QButtonGroup menu_buttonGroup;
	QButtonGroup clock_source_buttonGroup;
	QButtonGroup RCM_buttonGroup;
	QButtonGroup modbus_buttonGroup;
	QButtonGroup ethercat_buttonGroup;
    QButtonGroup power_control_buttonGroup;
	QButtonGroup PWM_triggering_buttonGroup;
	QButtonGroup Analog_input_buttonGroup;
	QButtonGroup External_triggering_buttonGroup;
	QButtonGroup logging_buttonGroup;

	int	last_page_index;		//Track the last menu visited for some miscellaneous use-cases
	int	power_menu_action;		//Choose Shutdown / Restart / Exit

	/* NUMPAD */
	/* target_parameter and target_button can be used to 'dynamically' refer to the most recently pressed 'control button' and accompanying relevant variable, on any of the pages.
	 * precision is used to decide how many decimals will be used.
	 * multiplier is for swapping between for example Hz and MHz.
	 * The contents of these four can be changed by using the numpad_preparation function below. */
	Numpad *numpad;
	QPushButton *target_parameter_button;
	double scale_multiplier;
	int precision;

    void numpad_parameter_select(QPushButton *button, double *parameter, QString unit, int precision = 2, double scale = 1, bool period_enable = true, bool sign_enable = false);
	void ok_check_minmax(QPushButton *button);		//Check if the value fulfill the min/max requirements of an input.
};

#endif //GUI_1CHANNEL_H
